﻿IncludeFile "SHA1.pb"

; ********* CONSTANTS *********
#ProgName                      = "PS3P_PKG_Ripper"
#Version                       = "V1.3"
#Debug                         = #False

#AesKeyLength                  = $10
#DebugKeyLength                = $40
#ChunkSize                     = $200000

Enumeration
  #Extract_Single
  #Extract_Template
EndEnumeration

Enumeration
  #PKG_File
  #Decrypted_File
EndEnumeration

Enumeration
  #Success
  #Error_PKG_Open
  #Error_Wrong_Magic
  #Error_Output_Dir_Creation
  #Error_No_Items
EndEnumeration

#PKG_Magic                     = $7F504B47
Enumeration
  #PKG_Revision_Debug
  #PKG_Revision_Retail
EndEnumeration
#PKG_Content_Type_PS3          = $0001
#PKG_Content_Type_PSP          = $0002

#PKG_Info_SizeOfContent_Offset = $0004

#ITEM_Content_Type_PS3         = $8000
#ITEM_Content_Type_PSP         = $9000
#ITEM_Type_Directory           = $0004

Enumeration
  #Buffer1
  #Buffer2
EndEnumeration



; ******* TYP DEFINITIONS ********
Structure PKG_Header
  Magic.l                           ; #PKG_Magic 
  Revision.w                        ; #PKG_Revision_Retail OR #PKG_Revision_Debug
  Type.w                            ; #PKG_Content_Type_PS3 OR #PKG_Content_Type_PSP
  Info_Offset.l
  Info_Count.l
  Info_Size.l
  Item_Count.l                      ; uiFileChunks
  Total_Size.q
  Data_Offset.q                     ; uiEncryptedFileStartOffset
  Data_Size.q                       ; uiEncryptedFileLenght
  ContentID.a[$24]
  Padding.a[$0C]
  Digest.a[$10]
  K_Licensee.a[$10]                 ; incPKGFileKey
  Header_CMAC_Hash.a[$10]
  Header_NPDRM_Signature.a[$28]
  Header_SHA1_Hash.a[$08]
EndStructure

Structure PKG_Info
  Packet_Identifier.l
  ContentSize.l
  *Content
EndStructure

Structure ITEM_Header
  Name_Offset.l
  Name_Size.l
  Data_Offset.q
  Data_Size.q
  Content_Type.u                    ; #ITEM_Content_Type_PS3 OR #ITEM_Content_Type_PSP
  Type.u                            ; #ITEM_Type_Directory / Any other value is a file(!)
  Padding.l
EndStructure

; ******* VARIABLES *******

Global NewList ProgramParameter$()
Global NewList ItemsToExtract$()
Global OutputDir$="", ListItems=#False, ExtractMode=#Extract_Single ; default values

Global PKG.PKG_Header
Global ITEM.ITEM_Header

Global *AesKey           = #Null
Global *Key              = AllocateMemory(#ChunkSize*4)
Global *KeyInUse         = #Null
Global KeyInUse_Length   = 0
Global TypeMultiplicator = 0 ; 1 for Retail 4 for Debug
Global *XOR_Key          = AllocateMemory(#ChunkSize)

Global *EncryptedData    = AllocateMemory(#ChunkSize)
Global Dim *DecryptedData(1) ; #Buffer1 and #Buffer1... one for write to file and one for decryt
*DecryptedData(#Buffer1)  = AllocateMemory(#ChunkSize) : *DecryptedData(#Buffer2) = AllocateMemory(#ChunkSize)
Global DecryptionBuffer  = #Buffer1, FileWriteBuffer = #Buffer2

; ******** DATA ********  
  
DataSection
  PS3AesKey:
  Data.b $2E,$7B,$71,$D7,$C9,$C9,$A1,$4E,$A3,$22,$1F,$18,$88,$28,$B8,$F8
  
  PSPAesKey:
  Data.b $07,$F2,$C6,$82,$90,$B5,$0D,$2C,$33,$81,$8D,$70,$9B,$60,$E6,$2B
EndDataSection


; ********* PROCEDURES ********

Procedure Create_Directory(Directory$)                ; Create directory chain
  If FileSize(Directory$) = -2 : ProcedureReturn #True : EndIf ; directory already exist
  Directory$=RTrim(ReplaceString(Directory$, "/","\"),"\")
  Dirs=CountString(Directory$, "\")+1
  NewDirectory$=""
  For i=1 To Dirs
    NewDirectory$+StringField(Directory$, i, "\")+"\"
    If FileSize(NewDirectory$)<>-2
      If CreateDirectory(NewDirectory$)=0 : ProcedureReturn #False : EndIf
    EndIf
  Next
  ProcedureReturn #True
EndProcedure

Procedure PrintUsage()
  PrintN("")
  PrintN(#ProgName+" "+#Version) : PrintN("")
  PrintN("Credits to: @mathieulh, @Mysis, @Ifcaro")
  PrintN("")
  PrintN("Usage:")
  PrintN(#ProgName+".exe [-o OutputDir] [-h|-l|-s File|-a Template|-f ItemsListFile] PKG_File") : PrintN("")
  PrintN("[-o OutputDir]     : Output Directory Path (Default = '.\[ContentID]')")
  PrintN("[-h]               : Usage Message")
  PrintN("[-l]               : Just List Files")
  PrintN("[-s File]          : Extract Single File (eg '-i USRDIR/EBOOT.BIN')")
  PrintN("[-a Template]      : Extract All Items Containing The Template (eg '-a .SFO')")
  PrintN("[-f ItemsListFile] : File Containing a List Of Items To Extract") : PrintN("")
  PrintN("NOTE: Use Quotes If A Path Contains Spaces")
EndProcedure

Procedure WriteBufferToFile(*Size)
  WriteData(#Decrypted_File, *DecryptedData(FileWriteBuffer), *Size)
EndProcedure


; Convert to String
Procedure.s MemToString(*Memory, Bytes)
  String$=""
  For i=0 To Bytes-1
    String$+Chr(PeekA(*Memory+i))
  Next
  ProcedureReturn String$
EndProcedure

CompilerIf #Debug
  Procedure.s MemToHexString(*Memory, Bytes)
    String$=""
    For i=0 To Bytes-1
      String$+RSet(Hex(PeekA(*Memory+i)),2,"0")
    Next
    ProcedureReturn String$
  EndProcedure
  
  Procedure.s ValueToHexString(Value, Type=#PB_Quad)
    Select Type
      Case #PB_Byte : ProcedureReturn RSet(Hex(Value, #PB_Byte), 2, "0")
      Case #PB_Word : ProcedureReturn RSet(Hex(Value, #PB_Word), 4, "0")
      Case #PB_Long : ProcedureReturn RSet(Hex(Value, #PB_Long), 8, "0")
      Case #PB_Quad : ProcedureReturn RSet(Hex(Value, #PB_Quad), 16, "0")
    EndSelect
  EndProcedure
  
  
  Procedure PrintPKGHeader()
    Debug "Magic: "+ValueToHexString(PKG\Magic, #PB_Long)
    Debug "PKG-Revision: "+ValueToHexString(PKG\Revision, #PB_Word)
    Debug "PKG-Type: "+ValueToHexString(PKG\Type, #PB_Word)
    Debug "PKG-Info-Offset: "+ValueToHexString(PKG\Info_Offset, #PB_Long)
    Debug "PKG-Info-count: "+ValueToHexString(PKG\Info_Count, #PB_Long)
    Debug "Header-Size: "+ValueToHexString(PKG\Info_Size, #PB_Long)
    Debug "Item Count: "+ValueToHexString(PKG\Item_Count, #PB_Long)
    Debug "Total-size: "+ValueToHexString(PKG\Total_Size)
    Debug "Data-Offset: "+ValueToHexString(PKG\Data_Offset)
    Debug "Data-Size: "+ValueToHexString(PKG\Data_Size)
    Debug "Content-ID: "+MemToString(@PKG\ContentID, SizeOf(PKG\ContentID)) 
    Debug "Padding: "+MemToHexString(@PKG\Padding, SizeOf(PKG\Padding))
    Debug "Digest: "+MemToHexString(@PKG\Digest, SizeOf(PKG\Digest))
    Debug "PKG-Data-IV: "+MemToHexString(@PKG\K_Licensee, SizeOf(PKG\K_Licensee))
    Debug "Header-CMAC-Hash:"+MemToHexString(@PKG\Header_CMAC_Hash, SizeOf(PKG\Header_CMAC_Hash))
    Debug "Header-NPDRM-Signature: "+MemToHexString(@PKG\Header_NPDRM_Signature,SizeOf(PKG\Header_NPDRM_Signature))
    Debug "Header-SHA1-Hash: "+MemToHexString(@PKG\Header_SHA1_Hash,SizeOf(PKG\Header_SHA1_Hash))
  EndProcedure
CompilerEndIf

; Big Endian Operations
Procedure ConvertEndianMem(*Memory, Length)          ; Length must be a multiple of 2
  For i=0 To (Length>>1)-1
    Tmp.a=PeekA(*Memory+i) : PokeA(*Memory+i, PeekA(*Memory+Length-1-i)) : PokeA(*Memory+Length-1-i, Tmp)
  Next
EndProcedure

Procedure ReadBigEndianMem(*Source, *Target, Length) ; Length can be 1,2,4,8(Byte, Word, Long, Quad)
  Select Length
    Case 1   : PokeA(*Target, PeekA(*Source))
    Case 2   : PokeW(*Target, PeekW(*Source))
    Case 4   : PokeL(*Target, PeekL(*Source))
    Case 8   : PokeQ(*Target, PeekQ(*Source))
    Default : CopyMemory(*Source, *Target, Length)
  EndSelect
  ConvertEndianMem(*Target, Length)
EndProcedure
  
Procedure ReadBigEndianFile(Source, *Target, Length) ; Length must be a multiple of 2
  ReadData(Source, *Target, Length)
  ConvertEndianMem(*Target, Length)
EndProcedure


; Header Parsing
Procedure ParsePKGHeader(File)
  ReadBigEndianFile(File, @PKG\Magic, SizeOf(PKG\Magic))
  ReadBigEndianFile(File, @PKG\Revision, SizeOf(PKG\Revision)) : If PKG\Revision : PKG\Revision = #PKG_Revision_Retail : EndIf ; Change 0x8000 to 0x0001
  ReadBigEndianFile(File, @PKG\Type, SizeOf(PKG\Type))
  ReadBigEndianFile(File, @PKG\Info_Offset, SizeOf(PKG\Info_Offset))
  ReadBigEndianFile(File, @PKG\Info_Count, SizeOf(PKG\Info_Count))
  ReadBigEndianFile(File, @PKG\Info_Size, SizeOf(PKG\Info_Size))
  ReadBigEndianFile(File, @PKG\Item_Count, SizeOf(PKG\Item_Count))
  ReadBigEndianFile(File, @PKG\Total_Size, SizeOf(PKG\Total_Size))
  ReadBigEndianFile(File, @PKG\Data_Offset, SizeOf(PKG\Data_Offset))
  ReadBigEndianFile(File, @PKG\Data_Size, SizeOf(PKG\Data_Size))
  ReadData(File, @PKG\ContentID, SizeOf(PKG\ContentID))
  ReadData(File, @PKG\Padding, SizeOf(PKG\Padding))
  ReadData(File, @PKG\Digest, SizeOf(PKG\Digest))
  ReadData(File, @PKG\K_Licensee, SizeOf(PKG\K_Licensee))
  ReadData(File, @PKG\Header_CMAC_Hash, SizeOf(PKG\Header_CMAC_Hash))
  ReadData(File, @PKG\Header_NPDRM_Signature, SizeOf(PKG\Header_NPDRM_Signature))
  ReadData(File, @PKG\Header_SHA1_Hash, SizeOf(PKG\Header_SHA1_Hash))
EndProcedure

Procedure ParseItemHeader(*ItemInfo)
  ReadBigEndianMem(*ItemInfo+OffsetOf(ITEM_Header\Name_Offset), @ITEM\Name_Offset, 4)
  ReadBigEndianMem(*ItemInfo+OffsetOf(ITEM_Header\Name_Size),   @ITEM\Name_Size, 4)
  ReadBigEndianMem(*ItemInfo+OffsetOf(ITEM_Header\Data_Offset), @ITEM\Data_Offset, 8)
  ReadBigEndianMem(*ItemInfo+OffsetOf(ITEM_Header\Data_Size),   @ITEM\Data_Size, 8)
  ReadBigEndianMem(*ItemInfo+OffsetOf(ITEM_Header\Content_Type),@ITEM\Content_Type, 2)
  ReadBigEndianMem(*ItemInfo+OffsetOf(ITEM_Header\Type),        @ITEM\Type, 2)
EndProcedure

; Key Manipulations
Procedure MakeRetail_XOR_Key(KeySize)
  AESEncoder(*Key, *XOR_Key, Keysize, *AesKey, 128, #Null, #PB_Cipher_ECB)
EndProcedure

Procedure MakeDebug_XOR_Key(KeySize)
  KeySize-1
  For j=0 To KeySize Step 16
;     SHA1$ = SHA1Fingerprint(*Key+j<<2, #DebugKeyLength)
;     For i=1 To 32 Step 2
;       PokeA(*XOR_Key+j+(i>>1), Val("$"+Mid(SHA1$,i,2)))
;     Next
    SHA1_Hash(*Key+(j<<2), #DebugKeyLength, *SHA1_Hash)
    ! MOV edx, dword[p_SHA1_Hash]
    ! MOV ecx, dword[p_XOR_Key]
    ! ADD ecx, dword[p.v_j]
    ! MOV eax, [edx]
    ! BSWAP eax
    ! MOV [ecx], eax
    ! MOV eax, [edx+4]
    ! BSWAP eax
    ! MOV [ecx+4], eax
    ! MOV eax, [edx+8]
    ! BSWAP eax
    ! MOV [ecx+8], eax
    ! MOV eax, [edx+12]
    ! BSWAP eax
    ! MOV [ecx+12], eax
  Next
EndProcedure

Prototype MakeXOR_Key(KeySize)
Global MakeXOR_Key.MakeXOR_Key = #Null

Procedure AddToKey(*TargetKey, Value.l, KeyLength.l)
  
  ! MOV edx, dword[p.p_TargetKey]
  ! MOV ecx, dword[p.v_KeyLength]
  ! MOV ebx, dword[p.v_Value]
  ! ADD edx, ecx
  
  ! SUB edx, 4
  ! SUB ecx, 4
  ! MOV eax, [edx]
  ! BSWAP eax
  ! ADD eax, ebx
  ! BSWAP eax
  ! MOV [edx], eax
  ! JC l_addtokey_overflow
  ProcedureReturn
  
overflow:
  ! SUB edx, 4
  ! SUB ecx, 4
  ! JGE l_addtokey_cont
  ProcedureReturn
cont:
  ! MOV eax, [edx]
  ! BSWAP eax
  ! INC eax
  ! BSWAP eax
  ! MOV [edx], eax
  ! JZ l_addtokey_overflow
  ProcedureReturn
        
EndProcedure

Procedure MultiplyAndIncrementKey(*TargetKey, KeySize.l, DataSize.l)
  
  ! MOV edx, dword[p.p_TargetKey]                ; KeyPointer
  ! MOV edi, dword[p.v_KeySize]
  ! MOV ecx, edi
  ! MOV esi, dword[p.v_DataSize]
start:  
  ! SUB esi, ecx
  ! JLE l_multiplyandincrementkey_exit

  ! MOV ebx, edx
  ! ADD ebx, ecx
copydword:
  ! MOV eax, [edx]
  ! MOV [ebx], eax
  ! ADD edx, 4
  ! ADD ebx, 4
  ! SUB ecx, 4
  ! JG l_multiplyandincrementkey_copydword       ; copy not finished
  ! MOV ebx, edx                                 ; Save pointer for next copy
  ! MOV ecx, edi                                 ; Restore KeySize
  
  ! ADD edx, ecx                                 ; Set KeyPointer behind key
increment:
  ! SUB edx, 4                                   ; move backwards 
  ! SUB ecx, 4                                   ; KeySize-DwordSize
  ! JGE l_multiplyandincrementkey_incrementdword ; End of key not reached -> increment next dword
  ! MOV ecx, edi                                 ; Restore KeySize
  ! MOV edx, ebx                                 ; Restore pointer for next copy
  ! JMP l_multiplyandincrementkey_start
  
incrementdword:
  ! MOV eax, [edx]
  ! BSWAP eax
  ! INC eax
  ! BSWAP eax
  ! MOV [edx], eax
  ! JZ l_multiplyandincrementkey_increment       ; Overflow -> increment next dword
  ! MOV ecx, edi                                 ; Restore KeySize
  ! MOV edx, ebx                                 ; Restore pointer for next copy
  ! JMP l_multiplyandincrementkey_start
exit:        
EndProcedure

Procedure IncrementKey(*TargetKey, KeySize.l)
  
  ! MOV edx, dword[p.p_TargetKey]
  ! MOV ecx, dword[p.v_KeySize]
  ! ADD edx, ecx
  
incrementstart:
  ! SUB edx, 4
  ! SUB ecx, 4
  ! JGE l_incrementkey_incrementdword
  ProcedureReturn
incrementdword:
  ! MOV eax, [edx]
  ! BSWAP eax
  ! INC eax
  ! BSWAP eax
  ! MOV [edx], eax
  ! JZ l_incrementkey_incrementstart
  ProcedureReturn
        
EndProcedure

Procedure CloneRetailKey(*SourceKey)
  PokeQ(*Key, PeekQ(*SourceKey))
  PokeQ(*Key+$08, PeekQ(*SourceKey+$08))
EndProcedure

Procedure CloneDebugKey(*SourceKey)
  FillMemory(*Key, #DebugKeyLength)
  
  PokeQ(*Key, PeekQ(*SourceKey))
  PokeQ(*Key+$08, PeekQ(*SourceKey))
  PokeQ(*Key+$10, PeekQ(*SourceKey+$08))
  PokeQ(*Key+$18, PeekQ(*SourceKey+$08))
EndProcedure

Prototype CloneKey(*SourceKey)
Global CloneKey.CloneKey=#Null

; Decrypt Engine
Procedure DecryptData(*Source, *Target, DataSize)
  
  MultiplyAndIncrementKey(*Key, KeyInUse_Length, DataSize*TypeMultiplicator)
  MakeXOR_Key(DataSize)
  
  ! MOV edx, dword[p.p_Target]
  ! MOV ecx, dword[p.p_Source]
  ! MOV ebx, [p_XOR_Key]
  ! MOV esi, dword[p.v_DataSize]
loop:
  ! SUB esi, 4
  ! JL l_decryptdata_exit
  ! MOV eax, [ecx]
  ! XOR eax, [ebx]
  ! MOV [edx], eax
  ! ADD edx, 4
  ! ADD ecx, 4
  ! ADD ebx, 4
  ! JMP l_decryptdata_loop
exit:
  
;   *TargetEnd=*Target+DataSize
;   *XORKey=*XOR_Key
;   While *Target < *TargetEnd
;     PokeL(*Target, (PeekL(*Source) ! PeekL(*XORKey)))
;     *Target+4 : *Source+4 : *XORKey+4
;   Wend
  
EndProcedure

Procedure DecryptChunk(ChunkSize, DataOffset.q, ItemDataOffset.q, PKG_File_Handle)
  
  ChunkSize=((ChunkSize+$0F) >> 4) << 4 ; align
  
  ; Read Chunk
  FileSeek(PKG_File_Handle, DataOffset + ItemDataOffset) : ReadData(PKG_File_Handle, *EncryptedData, ChunkSize)
  
  ; Increment Key
  CloneKey(*KeyInUse) : AddToKey(*Key, DataOffset>>4, KeyInUse_Length)
  
  DecryptData(*EncryptedData, *DecryptedData(DecryptionBuffer), ChunkSize)
  
EndProcedure

Procedure DecryptPKG(PKGFile$)
  
  ; *** Read PKG Header ***
  
  If ReadFile(#PKG_File, PKGFile$) = 0 : ProcedureReturn #Error_PKG_Open : EndIf
  
  ; Read PKG Header to PKG.PKG_Header
  ParsePKGHeader(#PKG_File)
  
  ; Check if valid PKG file
  If PKG\Magic<>#PKG_Magic : CloseFile(#PKG_File) : ProcedureReturn #Error_Wrong_Magic : EndIf
  
  PrintN("ContentID: "+PeekS(@PKG\ContentID)) : PrintN("")
  
  ; Create Output Directory
  If ListItems=#False
    If OutputDir$="" : OutputDir$=".\"+PeekS(@PKG\ContentID)+"\" : EndIf                       ; Default OutputDir is ContentID
    ReplaceString(OutputDir$, "/", "\") : If Right(OutputDir$,1)<>"\" : OutputDir$+"\" : EndIf ; Replace "/" with "\" and make sure "\" is at the end
    If Create_Directory(OutputDir$)=#False : CloseFile(#PKG_File) : ProcedureReturn #Error_Output_Dir_Creation : EndIf
  EndIf
  
  
  ; *** Decrypt Items Headers ***
  
  ; Set filepointer to start of encrypted items headers
  FileSeek(#PKG_File, PKG\Data_Offset)
  
  ; Allocate memory for items headers ('number of items'*'size of item header' (32 Bytes))
  ItemHeadersSize         = PKG\Item_Count << 5 ; * SizeOf(ITEM_Header)($20)
  If ItemHeadersSize = 0 : CloseFile(#PKG_File) : ProcedureReturn #Error_No_Items : EndIf
  OverallDataSize.q       = ItemHeadersSize
  *EncryptedItems         = AllocateMemory(ItemHeadersSize)
  *DecryptedItems         = AllocateMemory(ItemHeadersSize)
  
  ; Read encryped items headers  
  ReadData(#PKG_File, *EncryptedItems, ItemHeadersSize)
  
  
  ; *** Decrypt Items Headers ***
  
  ; Set procedure calls and variables regarding to retail/debug pkg
  CloneKey.CloneKey=@CloneRetailKey(): MakeXOR_Key.MakeXOR_Key=@MakeRetail_XOR_Key() : *KeyInUse=@PKG\K_Licensee : KeyInUse_Length=#AesKeyLength :TypeMultiplicator=1
  If PKG\Revision=#PKG_Revision_Debug : CloneKey.CloneKey=@CloneDebugKey() : MakeXOR_Key.MakeXOR_Key=@MakeDebug_XOR_Key() : *KeyInUse=@PKG\Digest : KeyInUse_Length=#DebugKeyLength : TypeMultiplicator=4 : EndIf
 
  ; Set keys and decrypt items headers
  CloneKey(*KeyInUse) : *AesKey=?PS3AesKey : If PKG\Type = #PKG_Content_Type_PSP : *AesKey=?PSPAESKey : EndIf
  DecryptData(*EncryptedItems, *DecryptedItems, ItemHeadersSize) ; ITEM_Headers
  
  OverallDataSize+ItemHeadersSize
  
  ; Caluculate and Allocate Memory Needed For Items Names
  ReadBigEndianMem(*DecryptedItems, @FirstNameOffset, 4) ; same as ItemHeadersSize !
  ReadBigEndianMem(*DecryptedItems+((PKG\Item_Count-1)<<5)+OffsetOf(ITEM_Header\Name_Offset), @LastNameOffset,4)
  ReadBigEndianMem(*DecryptedItems+((PKG\Item_Count-1)<<5)+OffsetOf(ITEM_Header\Name_Size), @LastNameSize,4)
  ItemsNamesSize=LastNameOffset+(((LastNameSize+$0f) >> 4) << 4) - FirstNameOffset : OverallDataSize+ItemsNamesSize
  *EncryptedItemsNames = AllocateMemory(ItemsNamesSize) : ReadData(#PKG_File, *EncryptedItemsNames, ItemsNamesSize)
  *DecryptedItemsNames = AllocateMemory(ItemsNamesSize)
  
  ; Decrypt Items Loop
  ItemsNamesSize=0
  For i=0 To PKG\Item_Count-1
    
    ; Read Item Header
    ParseItemHeader(*DecryptedItems+(i<<5))
    
    ;*** Decrypt Item Name ***
    ItemNameLength.l = ITEM\Name_Size : ItemNameLength = ((ItemNameLength+$0F) >> 4) << 4
    *AesKey=?PS3AESKey : If ITEM\Content_Type = #ITEM_Content_Type_PSP : *AesKey=?PSPAESKey : EndIf
    CloneKey(*KeyInUse) : AddToKey(*Key, (ITEM\Name_Offset)>>4, KeyInUse_Length)
    DecryptData(*EncryptedItemsNames+ItemsNamesSize, *DecryptedItemsNames+ItemsNamesSize, ItemNameLength)
    ItemsNamesSize+ItemNameLength
  

    ;*** Decrypt Item Data ***
    ; Get Item's Name
    ItemName$=PeekS(*DecryptedItemsNames+ITEM\Name_Offset-ItemHeadersSize, ITEM\Name_Size)
    
    ; Only extract selected Items
    If ListSize(ItemsToExtract$()) ; ListSize=0 -> no items are selected -> extract whole pkg 
      ItemNotFound=#True
      If ExtractMode=#Extract_Template
        ForEach ItemsToExtract$() : If FindString(ItemName$, ItemsToExtract$(), 1, #PB_String_NoCase) : ItemNotFound=#False : Break : EndIf : Next
      Else
        ForEach ItemsToExtract$() : If UCase(ItemName$)=UCase(ItemsToExtract$()) : ItemNotFound=#False : Break : EndIf : Next
      EndIf
      If ItemNotFound : Continue : EndIf ; skip item
    EndIf   
    
    ; Print Item Name if its not a directory
    If ITEM\Type<>#ITEM_Type_Directory
      Out$="PS3 File : " : If ITEM\Content_Type = #ITEM_Content_Type_PSP : Out$="PSP File : " : EndIf : Print(Out$+ItemName$)
    EndIf

    ; Only list items names
    If ListItems
      If ITEM\Type = #ITEM_Type_Directory : Continue : EndIf
      If ITEM\Data_Size<$400 : PrintN(" ("+Str(ITEM\Data_Size)+" Byte)") : Continue : EndIf
      If ITEM\Data_Size<$100000 : PrintN(" ("+StrD(ITEM\Data_Size/1024.0, 2)+" KB)") : Continue : EndIf
      If ITEM\Data_Size<$40000000 : PrintN(" ("+StrD(ITEM\Data_Size/1048576.0,2)+" MB)") : Continue : EndIf
      PrintN(" ("+StrD(ITEM\Data_Size/1073741824.0)+" GB)") : Continue
    EndIf
    
    ; If Item is 'Directory' -> Create Directory and continue Item Loop because Directory has no data to decrypt
    If ITEM\Type = #ITEM_Type_Directory : Create_Directory(OutputDir$+ItemName$) : Continue : EndIf
    PrintN("")
     
    ; From here on item is a file
    Create_Directory(GetPathPart(OutputDir$+ItemName$))
    
    ; *** Decrypt and Extract File ***
    ; Decrypt and save file in chunks
    CreateFile(#Decrypted_File, OutputDir$+ItemName$)
    
    ; Set AES key
    *AesKey = ?PS3AesKey : If ITEM\Content_Type = #ITEM_Content_Type_PSP : *AesKey=?PSPAesKey : EndIf
    
    ; File Chunks Loop
    FileDataSize.q = 0 ; used for progress calculation
    CurrentChunkSize.q=#ChunkSize
    Item_Offset.q = 0
    While Item_Offset<=ITEM\Data_Size-1
          
      ; Check if it's last chunk -> Change chunk size for save
      If ITEM\Data_Size-Item_Offset<#ChunkSize And ITEM\Data_Size % #ChunkSize : CurrentChunkSize = ITEM\Data_Size % #ChunkSize : EndIf
  
      ; Decrypt Chunk
      DecryptChunk(CurrentChunkSize, ITEM\Data_Offset+Item_Offset, PKG\Data_Offset, #PKG_File)
       
      ; Save decrypted chunk through write thread so decryption can continue while writing
      WaitThread(WriteThread) ; Wait for write thread to be finished
      Swap DecryptionBuffer, FileWriteBuffer ; swap buffers
      WriteThread=CreateThread(@WriteBufferToFile(), CurrentChunkSize) ; start new write thread
      ;WriteData(#Decrypted_File, *DecryptedData(DecryptionBuffer), CurrentChunkSize)                                      
      
      ; Output Progress
      FileDataSize+CurrentChunkSize : OverallDataSize+CurrentChunkSize
      PrintN(RSet(Str((OverallDataSize*100)/PKG\Data_Size),3, "0")+"% / "+RSet(Str((FileDataSize * 100)/ITEM\Data_Size), 3, "0")+"%")
      
      Item_Offset+CurrentChunkSize
    Wend
    
    WaitThread(WriteThread)
    CloseFile(#Decrypted_File) ; Close decrypted file 
    
  Next
    
  ; Clean up
  CloseFile(#PKG_File)
  FreeMemory(*EncryptedItems) : FreeMemory(*EncryptedItemsNames) : FreeMemory(*DecryptedItems) : FreeMemory(*DecryptedItemsNames)
  ProcedureReturn #Success
  
EndProcedure

; ******** MAIN ********

OpenConsole()

CompilerIf #debug
  OutputDir$="" : ListItems=#False
  ;EncryptedPKGFile$ = ".\test.pkg"
  EncryptedPKGFile$ = ".\Child Of Light [NPEB01934](Unlock).pkg"
  DecryptPKG(EncryptedPKGFile$)
CompilerElse

  ; Get Parameters from Console
  ProgramParameterCount=CountProgramParameters()
  For i = 1 To ProgramParameterCount : AddElement(ProgramParameter$()) : ProgramParameter$()=ProgramParameter() : Next
  If ListSize(ProgramParameter$())=0 : PrintUsage() : Goto Exit : EndIf 
  
  ; Parse Parameters
  ForEach ProgramParameter$()
    
    If Left(ProgramParameter$(),1)="-" ; SWITCH
      Select Mid(ProgramParameter$(),2,1)
        Case "h", "H" : PrintUsage()  : Goto Exit
          
        Case "l", "L" : ListItems=#True : Continue
          
        Case "s", "S" :
          If NextElement(ProgramParameter$()) : AddElement(ItemsToExtract$()) : ItemsToExtract$()=RTrim(ReplaceString(ProgramParameter$(), "\", "/"),"/") : ExtractMode=#Extract_Single : Continue : EndIf
          PrintUsage() : Goto Exit  
          
        Case "a", "A"
          If NextElement(ProgramParameter$()) : AddElement(ItemsToExtract$()) : ItemsToExtract$()=RTrim(ReplaceString(ProgramParameter$(), "\", "/"),"/") : ExtractMode=#Extract_Template : Continue : EndIf
          PrintUsage() : Goto Exit  
          
        Case "f", "F" :
          If NextElement(ProgramParameter$())
            If ReadFile(0, ProgramParameter$())
              While Eof(0) = 0
                Item$=RTrim(ReplaceString(ReadString(0), "\", "/"),"/") : If Item$<>"" : AddElement(ItemsToExtract$()) : ItemsToExtract$()=Item$ : EndIf
              Wend
            EndIf
            Continue
          EndIf
          PrintUsage() : Goto Exit 
          
        Case "o", "O" :
          If NextElement(ProgramParameter$()) : OutputDir$=ProgramParameter$() : Continue : EndIf
          PrintUsage() : Goto Exit
          
        Default : PrintUsage() : Goto Exit
      EndSelect
    Else
      EncryptedPKGFile$=ProgramParameter$() : If EncryptedPKGFile$="" : PrintUsage() : Goto Exit : EndIf
    EndIf
  Next
  
  PrintN("")
  PrintN("Summary:")
  PrintN("PKG-File           : "+EncryptedPKGFile$) : PrintN("")
  If OutputDir$<>"" : PrintN("OutPut Directory   : "+OutputDir$) : EndIf
  ForEach ItemsToExtract$()
    If ListIndex(ItemsToExtract$())=0
      PrintN("Item(s) To Extract : "+ItemsToExtract$())
    Else
      PrintN("                     "+ItemsToExtract$())
    EndIf
  Next
  If ListItems : PrintN("List Files         : TRUE") : EndIf
  PrintN("")
  
  StartTime=ElapsedMilliseconds()
  Select DecryptPKG(EncryptedPKGFile$)
    Case #Success                   : PrintN(Chr(10)+"Unpacking Successfully Finished")
    Case #Error_PKG_Open            : PrintN(Chr(10)+"Error: Could Not Open PKG-File '"+EncryptedPKGFile$+"'")
    Case #Error_Wrong_Magic         : PrintN(Chr(10)+"Error: Unsupported PKG-Type")
    Case #Error_Output_Dir_Creation : PrintN(Chr(10)+"Error: Could Not Create Output Directory")
    Case #Error_No_Items            : PrintN(Chr(10)+"Error: PKG Is Empty")
  EndSelect
  UsedTime$=Str((ElapsedMilliseconds()-StartTime)) : If Len(UsedTime$)<4 : RSet(UsedTime$, 4, "0") : EndIf
  PrintN("Time Used: "+InsertString(UsedTime$,".",Len(UsedTime$)-2)+" Seconds")
CompilerEndIf

Exit:
  FreeMemory(*DecryptedData(#Buffer1)) : FreeMemory(*DecryptedData(#Buffer2)) : FreeMemory(*EncryptedData) : FreeMemory(*SHA1_Hash)
  CloseConsole()
  



; IDE Options = PureBasic 5.11 (Windows - x86)
; ExecutableFormat = Console
; CursorPosition = 152
; FirstLine = 125
; Folding = EAI9
; EnableThread
; Executable = PS3P_PKG_Ripper.exe
; CPU = 1